<?php

defined('BASEPATH') OR exit('No direct script access allowed');

/* * *****************Newsletter.php**********************************
 * @product name    : Global Multi School Management System Express
 * @type            : Class
 * @class name      : Newsletter
 * @description     : Manage school academic newsletter.  
 * @author          : Codetroopers Team 	
 * @url             : https://themeforest.net/user/codetroopers      
 * @support         : yousuf361@gmail.com	
 * @copyright       : Codetroopers Team	 	
 * *********************** *********************************** */

class Newsletter extends MY_Controller {

    public $data = array();
    
    
    function __construct() {
        parent::__construct();
         $this->load->model('Newsletter_Model', 'newsletter', true);               
    }

    
    
    /*****************Function index**********************************
    * @type            : Function
    * @function name   : index
    * @description     : Load "Newsletter List" user interface                 
    *                        
    * @param           : null
    * @return          : null 
    * ********************************************************** */
    public function index($school_id = null) {
        
        check_permission(VIEW);

        $this->data['newsletter_list'] = $this->newsletter->get_newsletter_list($school_id); 
        $this->data['filter_school_id'] = $school_id;
        $this->data['schools'] = $this->schools;
        
        $this->data['list'] = TRUE;
        $this->layout->title( $this->lang->line('manage_newsletter'). ' | ' . SMS);
        $this->layout->view('newsletter/index', $this->data);            
       
    }

    
    /*****************Function add**********************************
    * @type            : Function
    * @function name   : add
    * @description     : Load "Add new Newsletter" user interface                 
    *                    and store "Newsletter" into database 
    * @param           : null
    * @return          : null 
    * ********************************************************** */
    public function add() {

        check_permission(ADD);
         
        if ($_POST) {
            $this->_prepare_newsletter_validation();
            if ($this->form_validation->run() === TRUE) {
                $data = $this->_get_posted_newsletter_data();

                $insert_id = $this->newsletter->insert('newsletter', $data);
                if ($insert_id) {
                    
                    create_log('Has been created a newsletter : '.$data['title']);
                    success($this->lang->line('insert_success'));
                    redirect('announcement/newsletter/index/'.$data['school_id']);
                    
                } else {
                    error($this->lang->line('insert_failed'));
                    redirect('announcement/newsletter/add');
                }
            } else {
                error($this->lang->line('insert_failed'));
                $this->data['post'] = $_POST;
            }
        }
        
        $this->data['newsletter_list'] = $this->newsletter->get_newsletter_list(); 
        $this->data['schools'] = $this->schools;
         
        $this->data['add'] = TRUE;
        $this->layout->title($this->lang->line('add'). ' | ' . SMS);
        $this->layout->view('newsletter/index', $this->data);
    }

    
    /*****************Function edit**********************************
    * @type            : Function
    * @function name   : edit
    * @description     : Load Update "Newsletter" user interface                 
    *                    with populated "Newsletter" value 
    *                    and update "Newsletter" database    
    * @param           : $id integer value
    * @return          : null 
    * ********************************************************** */
    public function edit($id = null) {       
       
         check_permission(EDIT);
        
         if(!is_numeric($id)){
            error($this->lang->line('unexpected_error'));
            redirect('announcement/newsletter/index');    
        }
        
        if ($_POST) {
            $this->_prepare_newsletter_validation();
            if ($this->form_validation->run() === TRUE) {
                $data = $this->_get_posted_newsletter_data();
                $updated = $this->newsletter->update('newsletter', $data, array('id' => $this->input->post('id')));

                if ($updated) {
                    
                     create_log('Has been updated a newsletter : '.$data['title']);
                    
                    success($this->lang->line('update_success'));
                    redirect('announcement/newsletter/index/'.$data['school_id']);                   
                } else {
                    error($this->lang->line('update_failed'));
                    redirect('announcement/newsletter/edit/' . $this->input->post('id'));
                }
            } else {
                error($this->lang->line('update_failed'));
                $this->data['newsletter'] = $this->newsletter->get_single('newsletter', array('id' => $this->input->post('id')));
            }
        }
        
        if ($id) {
            $this->data['newsletter'] = $this->newsletter->get_single('newsletter', array('id' => $id));

            if (!$this->data['newsletter']) {
                 redirect('announcement/newsletter/index');
            }
        }

        $this->data['newsletter_list'] = $this->newsletter->get_newsletter_list($this->data['newsletter']->school_id);  
        $this->data['school_id'] = $this->data['newsletter']->school_id;
        $this->data['filter_school_id'] = $this->data['newsletter']->school_id;
        $this->data['schools'] = $this->schools;
        
        $this->data['edit'] = TRUE;       
        $this->layout->title($this->lang->line('edit') . ' | ' . SMS);
        $this->layout->view('newsletter/index', $this->data);
    }
    
    
    /*****************Function view**********************************
    * @type            : Function
    * @function name   : view
    * @description     : Load user interface with specific newsletter data                 
    *                       
    * @param           : $id integer value
    * @return          : null 
    * ********************************************************** */
    public function view($id = null){
        
         check_permission(VIEW);
         if(!is_numeric($id)){
            error($this->lang->line('unexpected_error'));
            redirect('announcement/newsletter/index');    
        }
        
        $this->data['newsletter_list'] = $this->newsletter->get_newsletter_list(); 
        
        $this->data['newsletter'] = $this->newsletter->get_single('newsletter', array('id' => $id));
        $this->data['detail'] = TRUE;       
        $this->layout->title($this->lang->line('view'). ' ' . $this->lang->line('newsletter'). ' | ' . SMS);
        $this->layout->view('newsletter/index', $this->data);
    }

                   
     /*****************Function get_single_newsletter**********************************
     * @type            : Function
     * @function name   : get_single_newsletter
     * @description     : "Load single newsletter information" from database                  
     *                    to the user interface   
     * @param           : null
     * @return          : null 
     * ********************************************************** */
    public function get_single_newsletter(){
        
       $newsletter_id = $this->input->post('newsletter_id');
       
       $this->data['newsletter'] = $this->newsletter->get_single_newsletter($newsletter_id);
       echo $this->load->view('newsletter/get-single-newsletter', $this->data);
    }
    
    /*****************Function _prepare_newsletter_validation**********************************
    * @type            : Function
    * @function name   : _prepare_newsletter_validation
    * @description     : Process "Newsletter" user input data validation                 
    *                       
    * @param           : null
    * @return          : null 
    * ********************************************************** */
    private function _prepare_newsletter_validation() {
        $this->load->library('form_validation');
        $this->form_validation->set_error_delimiters('<div class="error-message" style="color: red;">', '</div>');
        
        $this->form_validation->set_rules('school_id', $this->lang->line('school_name'), 'trim|required');   
        $this->form_validation->set_rules('title', $this->lang->line('title'), 'trim|required|callback_title');   
        $this->form_validation->set_rules('date', $this->lang->line('date'), 'trim|required');   
        $this->form_validation->set_rules('newsletter', $this->lang->line('newsletter'), 'trim|required');   
           
    }
   
    
    /*****************Function title**********************************
    * @type            : Function
    * @function name   : title
    * @description     : Unique check for "newsletter title" data/value                  
    *                       
    * @param           : null
    * @return          : boolean true/false 
    * ********************************************************** */ 
   public function title()
   {             
      if($this->input->post('id') == '')
      {   
          $newsletter = $this->newsletter->duplicate_check($this->input->post('school_id'),$this->input->post('title'), $this->input->post('date')); 
          if($newsletter){
                $this->form_validation->set_message('title', $this->lang->line('already_exist'));         
                return FALSE;
          } else {
              return TRUE;
          }          
      }else if($this->input->post('id') != ''){   
         $newsletter = $this->newsletter->duplicate_check($this->input->post('school_id'), $this->input->post('title'),$this->input->post('date'), $this->input->post('id')); 
          if($newsletter){
                $this->form_validation->set_message('title', $this->lang->line('already_exist'));         
                return FALSE;
          } else {
              return TRUE;
          }
      }   
   }

   
   
    /*****************Function image**********************************
* @type            : Function
* @function name   : image
* @description     : Check file format validation                  
*                       
* @param           : null
* @return          : boolean true/false 
* ********************************************************** */ 
public function image()
{   
    if ($_FILES['image']['name']) {
        $name = $_FILES['image']['name'];
        $ext = strtolower(pathinfo($name, PATHINFO_EXTENSION)); // Convert extension to lowercase for uniformity
        
        // No need to check image dimensions
        // if ((!empty($width)) && ($width > 750 || $height > 500)) {
        //     $this->form_validation->set_message('image', $this->lang->line('please_check_image_dimension'));
        //     return FALSE;
        // }
        
        // Define allowed file extensions
        $allowed_extensions = array('jpg', 'jpeg', 'doc', 'docx', 'pdf', 'png', 'gif', 'xls', 'xlsx', 'ppt', 'pptx', 'txt', 'csv', 'zip', 'rar', '7z', 'mp3', 'mp4', 'avi', 'mov', 'wmv');
        
        // Check if the uploaded file extension is in the list of allowed extensions
        if (in_array($ext, $allowed_extensions)) {
            return TRUE;
        } else {
            $this->form_validation->set_message('image', $this->lang->line('select_valid_file_format'));         
            return FALSE; 
        }
    }       
}

/*****************Function _get_posted_newsletter_data**********************************
* @type            : Function
* @function name   : _get_posted_newsletter_data
* @description     : Prepare "Newsletter" user input data to save into the database                  
*                       
* @param           : null
* @return          : $data array(); value 
* ********************************************************** */
private function _get_posted_newsletter_data() {

    $items = array();
    $items[] = 'school_id';
    $items[] = 'title';
    $items[] = 'newsletter';
    $items[] = 'is_view_on_web';
    $data = elements($items, $_POST);  

    $data['date'] = date('Y-m-d', strtotime($this->input->post('date')));

    if ($this->input->post('id')) {
        $data['modified_at'] = date('Y-m-d H:i:s');
        $data['modified_by'] = logged_in_user_id();
    } else {
        $data['status'] = 1;
        $data['created_at'] = date('Y-m-d H:i:s');
        $data['created_by'] = logged_in_user_id();                       
    }

    if(isset($_FILES['image']['name'])){  
        $data['image'] =  $this->_upload_image();
    }

    return $data;
}

/*****************Function _upload_image**********************************
* @type            : Function
* @function name   : _upload_image
* @description     : Process to upload the newsletter image and return the image name                 
*                       
* @param           : null
* @return          : $return_image  string value 
* ********************************************************** */
private function _upload_image(){

    $prev_image     = $this->input->post('prev_image');
    $image          = $_FILES['image']['name'];
    $image_type     = $_FILES['image']['type'];
    $return_image   = '';
    if ($image != "") {
        // No need to check for image types
        // if ($image_type == 'image/jpeg' || $image_type == 'image/pjpeg' ||
        //         $image_type == 'image/jpg' || $image_type == 'image/png' ||
        //         $image_type == 'image/x-png' || $image_type == 'image/gif') {

        $destination = 'assets/uploads/newsletter/';               

        $file_type  = explode(".", $image);
        $extension  = strtolower($file_type[count($file_type) - 1]);
        $image_path = 'newsletter-'.time() . '-sms.' . $extension;

        move_uploaded_file($_FILES['image']['tmp_name'], $destination . $image_path);

        // need to unlink previous image
        if ($prev_image != "") {
            if (file_exists($destination . $prev_image)) {
                @unlink($destination . $prev_image);
            }
        }

        $return_image = $image_path;
    }
    // No need for an else statement
    // else {
    //     $return_image = $prev_image;
    // }

    return $return_image;
}

    
     /*****************Function delete**********************************
     * @type            : Function
     * @function name   : delete
     * @description     : delete "Newsletter" from database                  
     *                    and unlink newsletter image from server   
     * @param           : $id integer value
     * @return          : null 
     * ********************************************************** */
    public function delete($id = null) {
        
         check_permission(DELETE);
         
        if(!is_numeric($id)){
            error($this->lang->line('unexpected_error'));
            redirect('announcement/newsletter/index');    
        }  
        
        $newsletter = $this->newsletter->get_single('newsletter', array('id' => $id));
         
        if ($this->newsletter->delete('newsletter', array('id' => $id))) {  
            
            // delete teacher resume and image
            $destination = 'assets/uploads/';
            if (file_exists( $destination.'/newsletter/'.$newsletter->image)) {
                @unlink($destination.'/newsletter/'.$newsletter->image);
            }
            
            create_log('Has been deleted a newsletter : '.$newsletter->title); 
            
            success($this->lang->line('delete_success'));
        } else {
            error($this->lang->line('delete_failed'));
        }
        redirect('announcement/newsletter/index/'.$newsletter->school_id); 
    }

}
